<?php
/**
 * API endpoint to update redirect configuration
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Configuration files with absolute paths for Apache compatibility
$REDIRECT_URL_FILE = __DIR__ . '/redirect_url.txt';
$BOT_URL_FILE = __DIR__ . '/bot_url.txt';

try {
    // Get POST data
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input || !isset($input['type'])) {
        throw new Exception('Invalid request data');
    }
    
    $type = strtolower(trim($input['type']));
    
    // Handle different types of updates
    if ($type === 'apikey') {
        $value = trim($input['value']);
    } else {
        $url = trim($input['url']);
        // Validate URL only for URL types
        if (!filter_var($url, FILTER_VALIDATE_URL)) {
            throw new Exception('Invalid URL format');
        }
    }
    
    // Update appropriate file
    if ($type === 'human') {
        $result = file_put_contents($REDIRECT_URL_FILE, $url, LOCK_EX);
        if ($result === false) {
            $error = error_get_last();
            throw new Exception('Failed to save human redirect URL: ' . ($error['message'] ?? 'Unknown error'));
        }
        @chmod($REDIRECT_URL_FILE, 0644);
    } elseif ($type === 'bot') {
        $result = file_put_contents($BOT_URL_FILE, $url, LOCK_EX);
        if ($result === false) {
            $error = error_get_last();
            throw new Exception('Failed to save bot redirect URL: ' . ($error['message'] ?? 'Unknown error'));
        }
        @chmod($BOT_URL_FILE, 0644);
    } elseif ($type === 'apikey') {
        // Handle API key update
        $apiKey = trim($input['value']);
        if (empty($apiKey)) {
            throw new Exception('API key cannot be empty');
        }
        
        // Get absolute path for better Apache compatibility
        $apiKeyFile = __DIR__ . '/api_key.txt';
        
        // Check directory permissions
        if (!is_writable(__DIR__)) {
            throw new Exception('Directory not writable. Please check file permissions.');
        }
        
        // Save API key to file (for persistence across restarts)
        $result = file_put_contents($apiKeyFile, $apiKey, LOCK_EX);
        if ($result === false) {
            $error = error_get_last();
            throw new Exception('Failed to save API key: ' . ($error['message'] ?? 'Unknown error'));
        }
        
        // Set proper file permissions for Apache
        if (function_exists('chmod')) {
            @chmod($apiKeyFile, 0644);
        }
        
        // Return success response for API key
        echo json_encode([
            'success' => true,
            'message' => 'API key updated successfully',
            'type' => $type,
            'file_size' => $result
        ]);
        exit;
    } else {
        throw new Exception('Invalid type. Must be "human", "bot", or "apikey"');
    }
    
    // Return success response
    echo json_encode([
        'success' => true,
        'message' => ucfirst($type) . ' redirect URL updated successfully',
        'type' => $type,
        'url' => $url
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>